/*!
 * (C) Ionic http://ionicframework.com - MIT License
 */
import { Host, h } from "@stencil/core";
import { inheritAriaAttributes } from "../../utils/helpers";
import { createColorClasses, hostContext, openURL } from "../../utils/theme";
import { arrowBackSharp, chevronBack } from "ionicons/icons";
import { config } from "../../global/config";
import { getIonMode } from "../../global/ionic-global";
/**
 * @virtualProp {"ios" | "md"} mode - The mode determines which platform styles to use.
 *
 * @part native - The native HTML button element that wraps all child elements.
 * @part icon - The back button icon (uses ion-icon).
 * @part text - The back button text.
 */
export class BackButton {
    constructor() {
        this.inheritedAttributes = {};
        this.onClick = async (ev) => {
            const nav = this.el.closest('ion-nav');
            ev.preventDefault();
            if (nav && (await nav.canGoBack())) {
                return nav.pop({ animationBuilder: this.routerAnimation, skipIfBusy: true });
            }
            return openURL(this.defaultHref, ev, 'back', this.routerAnimation);
        };
        this.color = undefined;
        this.defaultHref = undefined;
        this.disabled = false;
        this.icon = undefined;
        this.text = undefined;
        this.type = 'button';
        this.routerAnimation = undefined;
    }
    componentWillLoad() {
        this.inheritedAttributes = inheritAriaAttributes(this.el);
        if (this.defaultHref === undefined) {
            this.defaultHref = config.get('backButtonDefaultHref');
        }
    }
    get backButtonIcon() {
        const icon = this.icon;
        if (icon != null) {
            // icon is set on the component or by the config
            return icon;
        }
        if (getIonMode(this) === 'ios') {
            // default ios back button icon
            return config.get('backButtonIcon', chevronBack);
        }
        // default md back button icon
        return config.get('backButtonIcon', arrowBackSharp);
    }
    get backButtonText() {
        const defaultBackButtonText = getIonMode(this) === 'ios' ? 'Back' : null;
        return this.text != null ? this.text : config.get('backButtonText', defaultBackButtonText);
    }
    get hasIconOnly() {
        return this.backButtonIcon && !this.backButtonText;
    }
    get rippleType() {
        // If the button only has an icon we use the unbounded
        // "circular" ripple effect
        if (this.hasIconOnly) {
            return 'unbounded';
        }
        return 'bounded';
    }
    render() {
        const { color, defaultHref, disabled, type, hasIconOnly, backButtonIcon, backButtonText, icon, inheritedAttributes, } = this;
        const showBackButton = defaultHref !== undefined;
        const mode = getIonMode(this);
        const ariaLabel = inheritedAttributes['aria-label'] || backButtonText || 'back';
        return (h(Host, { key: '9e837be314ee8ea487ba2771d36668ef58442faa', onClick: this.onClick, class: createColorClasses(color, {
                [mode]: true,
                button: true, // ion-buttons target .button
                'back-button-disabled': disabled,
                'back-button-has-icon-only': hasIconOnly,
                'in-toolbar': hostContext('ion-toolbar', this.el),
                'in-toolbar-color': hostContext('ion-toolbar[color]', this.el),
                'ion-activatable': true,
                'ion-focusable': true,
                'show-back-button': showBackButton,
            }) }, h("button", { key: 'c4531348bd80cf6163931c5fb432526aa8127043', type: type, disabled: disabled, class: "button-native", part: "native", "aria-label": ariaLabel }, h("span", { key: 'aab8c6d5f0a84aed9f4e4a2532a239b28c7d5589', class: "button-inner" }, backButtonIcon && (h("ion-icon", { key: '121eae10947b46a3d109709b79f4126b1c40ade4', part: "icon", icon: backButtonIcon, "aria-hidden": "true", lazy: false, "flip-rtl": icon === undefined })), backButtonText && (h("span", { key: '11c28875cc52b99470525d18c1f88f27c390a75a', part: "text", "aria-hidden": "true", class: "button-text" }, backButtonText))), mode === 'md' && h("ion-ripple-effect", { key: '804bf8a06cd033c99d61ec2665c842b3b25945fa', type: this.rippleType }))));
    }
    static get is() { return "ion-back-button"; }
    static get encapsulation() { return "shadow"; }
    static get originalStyleUrls() {
        return {
            "ios": ["back-button.ios.scss"],
            "md": ["back-button.md.scss"]
        };
    }
    static get styleUrls() {
        return {
            "ios": ["back-button.ios.css"],
            "md": ["back-button.md.css"]
        };
    }
    static get properties() {
        return {
            "color": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "Color",
                    "resolved": "\"danger\" | \"dark\" | \"light\" | \"medium\" | \"primary\" | \"secondary\" | \"success\" | \"tertiary\" | \"warning\" | string & Record<never, never> | undefined",
                    "references": {
                        "Color": {
                            "location": "import",
                            "path": "../../interface",
                            "id": "src/interface.d.ts::Color"
                        }
                    }
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The color to use from your application's color palette.\nDefault options are: `\"primary\"`, `\"secondary\"`, `\"tertiary\"`, `\"success\"`, `\"warning\"`, `\"danger\"`, `\"light\"`, `\"medium\"`, and `\"dark\"`.\nFor more information on colors, see [theming](/docs/theming/basics)."
                },
                "attribute": "color",
                "reflect": true
            },
            "defaultHref": {
                "type": "string",
                "mutable": true,
                "complexType": {
                    "original": "string",
                    "resolved": "string | undefined",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The url to navigate back to by default when there is no history."
                },
                "attribute": "default-href",
                "reflect": false
            },
            "disabled": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "If `true`, the user cannot interact with the button."
                },
                "attribute": "disabled",
                "reflect": true,
                "defaultValue": "false"
            },
            "icon": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string | null",
                    "resolved": "null | string | undefined",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The built-in named SVG icon name or the exact `src` of an SVG file\nto use for the back button."
                },
                "attribute": "icon",
                "reflect": false
            },
            "text": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string | null",
                    "resolved": "null | string | undefined",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The text to display in the back button."
                },
                "attribute": "text",
                "reflect": false
            },
            "type": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "'submit' | 'reset' | 'button'",
                    "resolved": "\"button\" | \"reset\" | \"submit\"",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "The type of the button."
                },
                "attribute": "type",
                "reflect": false,
                "defaultValue": "'button'"
            },
            "routerAnimation": {
                "type": "unknown",
                "mutable": false,
                "complexType": {
                    "original": "AnimationBuilder | undefined",
                    "resolved": "((baseEl: any, opts?: any) => Animation) | undefined",
                    "references": {
                        "AnimationBuilder": {
                            "location": "import",
                            "path": "../../interface",
                            "id": "src/interface.d.ts::AnimationBuilder"
                        }
                    }
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "When using a router, it specifies the transition animation when navigating to\nanother page."
                }
            }
        };
    }
    static get elementRef() { return "el"; }
}
